unit FTPFormTools;
{
    UNIT FTPFormTools;
    Version number 1.0(beta)

This unit contains tools (class TFTPFormTools) for main TFTPForm.

Created by Pter Karsai }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, ComCtrls, ToolWin, StdCtrls, Buttons, ExtCtrls, Grids, Winsock,
  WSocket, DirList, FTPSession, FTPCacheManager, SetupFormUnit,
  TransferFormUnit, Registry, ShellAPI;

{------------------------------------------------------------------------------}
{ caX* constants for AddToCommArea() }
const caLocal  = 0;
      caRemote = 1;
      caError  = 2;
      caMisc   = 3;

{ doX* constants for OnDConConnected and DataOperation }
const doList     = 0;  { receiving / list }
      doDownload = 1;  { receiving / file }
      doUpload   = 2;  { sending   / file }

{ Class TFTPForTools header declaration }
type TFTPFormTools = class(TObject)
{ ----------------------------------------------------------------------------}
{ Private miscellaneous methods }
{ ----------------------------------------------------------------------------}
private
   DataOperation   : byte;  { current operation }

   procedure AddToLocal(ICaption, IName, IDate: AnsiString; IIIndex,Data: word);
{ Function: Add a new entry to Local Browser }

   procedure AddToRemote(DirDetailRec: DirDetailRecord);
{ Function: Add a new entry to Remote Browser }

   procedure Delay(MSecs: longint);
{ Function: Wait MSecs milliseconds }

{ ----------------------------------------------------------------------------}
public
    LocalName   : string;  { local host name }
    LocalIP     : string;  { local IP for PORT command }
    LocalIPDot  : string;  { local dotted IP }

    ControlConnected : boolean; { if TRUE, control session alive }

    RemoteSite       : string;  { address or name of remote site }
    RemotePort       : word;    { port on remote host to connect to }
    UserName         : string;  { eh }
    Password         : string;  { ah }
    RetryOperation   : boolean; { TRUE if retry operation is in progress }

    DelaySpent  : boolean;
    CanContinue : boolean;      { global variable shows whether operation can
                                  continued... }
    CanResume   : boolean;      { TRUE if server support resuming downloads }

    XFTPSession : TFTPSession;   { FTP session manager }
    XFTPCache   : TFTPCache;     { FTP cache manager }

    CurrentRemPath : string;     { current remote path }
    SysTempPath    : string;     { system temp path }
    ControlIsFree  : boolean;    { TRUE if control connection is free for
                                   user operations }
{ ----------------------------------------------------------------------------}
{ General tools }
{ ----------------------------------------------------------------------------}
   procedure ShellExec(Command, DoWith: string);
{ Function: executes a ShellExecute() call with error handling. Parameter
  command is the operation you want to do with DoWith parameter (e.g. 'explore',
  'c:\') }

   procedure ShowError(ErrorMessage: string; var SetItToFalse: boolean);
{ Function: Show up an error message window. SetItToFalse parameter is a
  boolean... it's a frequent problem, canContinue... got it?}

   procedure AddToCommArea(Msg: string; MsgType: byte);
{ Function; Add a line (Msg) to CommArea richedit, MsgType can be caLocal,
  caRemote, caError (for different colors). }

   procedure RefreshLocalBrowser;
{ Function: refresh local browser folder data (current directory). }

   procedure RefreshRemoteBrowser(FolderFile: string);
{ Function: refresh remote browser folder data (current remote directory). }

   procedure ConnectToRemoteHost;
{ Function: try to connect to remote server and try to log in (connect settings
  as in Preferences set. }

   procedure Login;
{ Function: login if connection got. }

   function CanResumeDownloads: boolean;
{ Function: return TRUE if server support resuming broken downloads (REST). }

   function GetCurrentPath: string;
{ Function: return with the current path, if returning string is empty, an
  error occur. }

   procedure LoadDir(ForceReload: boolean);
{ Function: download remote directory or reload it from cache. Full cache.
  If ForceReload is TRUE, LoadDir() will download remote directory without
  using existing cache entries (but register the downloaded one). }

   procedure Download(OverWrite: boolean; RemoteFile, RemoteSize, LocalFile,
                      LocalSize: string);
{ Function: download RemoteFile to LocalFile }

   procedure Upload(OverWrite: boolean; RemoteFile, RemoteSize, LocalFile,
                    LocalSize: string);
{ Function: upload LocalFile to RemoteFile. }

   procedure SaveInitSettings;
{ Function: save all initial settings for YAFTP to system registry. }

   procedure LoadInitSettings;
{ Function: load all initial settings for YAFTP from system registry. }

   procedure RunGC;
{ Function: runs the starting garbage collector. Messages go to CommArea. }

{ ----------------------------------------------------------------------------}
{ "Event handlers" - external procedures }
{ ----------------------------------------------------------------------------}
   procedure OnCConConnected;  { called when control connection built }
   procedure OnCConCommand;    { called when a control command sent }
   procedure OnServerReply;    { called when a server reply arrive }
   procedure OnCConClosed;     { called when control connection closed }
   procedure OnCConError;      { called when an error occur in control con. }
   procedure OnDConConnected;  { called when data connection built }
   procedure OnDConClosed;     { called when data connection closed }
   procedure OnDConError;      { called when an error occur in data con.}
end;
{------------------------------------------------------------------------------}

implementation

uses FTPFormUnit;  { circular referencing for easier access }
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ Private miscellaneous methods -----------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TFTPFormTools.AddToLocal(ICaption, IName, IDate: AnsiString;
                                   IIIndex, Data: word);
var listItem: TListItem;  { to modify LocalList }
begin
{ add item }
    listItem            := FTPForm.LocalList.Items.Add;
{ set properties }
    listItem.Caption    := ICaption;
    listItem.ImageIndex := IIIndex;
    listItem.SubItems.Add(IName);
    listItem.SubItems.Add(IDate);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.AddToRemote(DirDetailRec: DirDetailRecord);
var listItem: TListItem;  { to modify RemoteList }
begin
{ add an item }
    listItem := FTPForm.RemoteList.Items.Add;
    with DirDetailRec do begin
    { set icon if it's a folder/link/file }
         case dd_Info of
              'F': listItem.ImageIndex:= iiFile;
              'D': listItem.ImageIndex:= iiNormalFolder;
              'L': listItem.ImageIndex:= iiLinkFolder;
              'P': listItem.ImageIndex:= iiParentFolder;
         end;
    { add other information }
         listItem.Caption := dd_Name;  { store name }
         if dd_Info = 'P' then
            listItem.Caption:= 'Parent folder'
         else begin
            listItem.SubItems.Add(IntToStr(dd_Size));
            listItem.SubItems.Add(dd_Date);
            listItem.SubItems.Add(dd_Rights);
            listItem.SubItems.Add(dd_Owner);
            listItem.SubItems.Add(dd_OwnerGroup);
            listItem.SubItems.Add(dd_LinksTo);
            listItem.SubItems.Add(IntToStr(dd_INodes));
         end;
    end;
end;

{------------------------------------------------------------------------------}

function CustomSortProc(Item1, Item2: TListItem; ParamSort: integer): integer;
                        stdcall;
const SameList: array[1..3] of word = (iiFile, iiNormalFolder, iiLinkFolder);
begin
{ files go behind folders }
  if (TListItem(Item1).ImageIndex = iiFile) and
     ((TListItem(Item2).ImageIndex = iiNormalFolder) or
      (TListItem(Item2).ImageIndex = iiParentFolder) or
      (TListItem(Item2).ImageIndex = iiLinkFolder))
  then
     Result:= 2
  else
     Result:= 0;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.Delay(MSecs: longint);
begin
{ setup WaitTimer }
    DelaySpent:= false;  { we've just started waiting... }
{ show the fact }
    AddToCommArea(Format('Delaying %d seconds...', [MSecs div 1000]), caMisc);
{ do it }
    with FTPForm.WaitTimer do begin
         Interval:= MSecs;
    { start timer }
         Enabled:= true;
    { wait 'till it finished }
         while not DelaySpent and canContinue do begin
               Application.ProcessMessages;
               if Application.Terminated then exit;
         end;
    {... now stop timer }
         Enabled:= false;
    end;  { good bye }
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ General tools ---------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TFTPFormTools.ShellExec(Command, DoWith: string);
const ErrorMsg = 'An error occured while attempting to execute a ' +
                 'shell call.'#10#10;
var retShell: longint;
begin
{ do ShellExecute() call }
    retShell:= ShellExecute(Application.Handle, PChar(Command), PChar(DoWith),
                            nil, nil, SW_SHOWNORMAL);
{ if there was an error, try to show error message }
    if retShell <= 32 then
       Application.MessageBox(PChar(ErrorMsg + '"' +
                                    FTPForm.ShellErrorMsg[retShell] + '"'),
                              'YAFTP information', mb_OK + mb_ICONEXCLAMATION);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.ShowError(ErrorMessage: string;
                                  var SetItToFalse: boolean);
begin
{ set that boolean to false }
    SetItToFalse:= false;
{ show error message }
    Application.MessageBox(PChar(ErrorMessage), 'YAFTP > Error',
                           mb_OK + mb_IconExclamation);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.AddToCommArea(Msg: string; MsgType: byte);
var colorIndex: integer;
    selAt     : integer;
begin
{ don't let list grow out 100 lines }
    if FTPForm.CommArea.Lines.Count >= 100 then
    begin
        selAt:= FTPForm.CommArea.SelStart;
        FTPForm.CommArea.Lines.Delete(0);
        FTPForm.CommArea.SelStart:= selAt;
    end;

{ choose color }
    case MsgType of
       caLocal  : colorIndex:= clGreen;
       caRemote : colorIndex:= clNavy;
       caError  : colorIndex:= clRed;
       caMisc   : colorIndex:= clMaroon;
    end;
{ add line if first line not eq disconnect sign line }
    if FTPForm.CommArea.Lines.Count = 1 then
       if (Copy(FTPForm.CommArea.Lines[0], 0, 2) = '><') then Exit;

    FTPForm.CommArea.SelAttributes.Color:= colorIndex;
    FTPForm.CommArea.Lines.Add(Msg);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.RefreshLocalBrowser;
var currentRec   : TSearchRec;  { for FindFirst(), FindNext(), FindClose() }
     fileFound   : integer;     { if 0, a file or folder found }
     fileDate    : TDateTime;   { conversion variable }
begin
{ clear locallist items }
     FTPForm.LocalList.Visible:= false;
     FTPForm.LocalList.Items.Clear;
     FTPForm.LocalList.Visible:= true;

{ add Parent directory }
     if Length(GetCurrentDir) > 3 then begin
        AddToLocal('Parent folder', '', '', iiParentFolder,0);
        FTPForm.LocalList.ItemFocused:= FTPForm.LocalList.Items[0];
     end;

{ add files or folders }
     if Length(GetCurrentDir) > 3 then  { e.g. 'c:\'. do not add \ !' }
        fileFound:= FindFirst(GetCurrentDir + '\*.*', faAnyFile, currentRec)
     else
        fileFound:= FindFirst(GetCurrentDir + '*.*', faAnyFile, currentRec);

     while fileFound = 0 do with currentRec do begin
     { filter '.' and '..' directories...}
         if not ((Name = '.') or (Name = '..')) then begin
         { convert file date }
            fileDate:= FileDateToDateTime(Time);
         { mask attribute: we have to mask because some directories are hidden,
           too, they don't have just faDirectory attribute. Assign ImageIndex. }
            if Attr and faDirectory = faDirectory then
               AddToLocal(Name, IntToStr(Size), DateTimeToStr(fileDate),
                          iiNormalFolder, 0)
            else
               AddToLocal(Name, IntToStr(Size), DateTimeToStr(fileDate),
                          iiFile, 1);
         end;
     { find next matching file }
         fileFound:= FindNext(currentRec);
     end;

{ free allocated Windows resources }
     FindClose(currentRec);

{ add directory name and path to LocalDirCombo }
     with FTPForm.LocalDirCombo do
     begin
     { we let 32 lines in the history }
         if Items.Count >= 32 then Items.Delete(0);
         if Items.IndexOf(GetCurrentDir) = -1 then
            ItemIndex:= Items.Add(GetCurrentDir);
     end;

{ sort items }
    FTPForm.LocalList.CustomSort(@CustomSortProc, 0);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.RefreshRemoteBrowser(FolderFile: string);
var tempDirList : TDirList;        { temporary DirList manager }
    tempDetail  : DirDetailRecord; { guess... }
    canContinue : boolean;         { guess... :)}
    lineCount   : longint;         { cyclevar }
begin
{ clear RemoteList items }
     FTPForm.RemoteList.Visible:= false;
     FTPForm.RemoteList.Items.Clear;
     FTPForm.RemoteList.Visible:= true;
{ try to create tempDirList }
     canContinue:= true;
     try
       tempDirList:= TDirList.Create(FolderFile);
     except
     { I/O error, e.g. file not found }
       on E:EInOutError do
          ShowError('Directory file access error: ' + E.Message, canContinue);
     { EInvalidFormat thrown if DirList couldn't determine file structure }
       on E:EInvalidFormat do
          ShowError(E.Message, canContinue);
     { ELineTooLong if line was too long... }
       on E:ELineTooLong do
          ShowError('Directory list error: ' + E.Message, canContinue);
     end;

{ if we can continue, just continue it...}
     if canContinue then
     begin
     { add parent directory entry }
        if CurrentRemPath <> '/' then
        begin
             tempDetail.dd_Info:= 'P'; AddToRemote(tempDetail);
        end;

     { add all info lines to RemoteBrowser }
        for lineCount:= 0 to tempDirList.GetLineCount - 1 do
        begin
            tempDirList.GetDetails(lineCount, tempDetail);  { read details... }
        { add details except they're up and current directory entries }
            if (tempDetail.dd_Name <> '..') and (tempDetail.dd_Name <> '.') then
               AddToRemote(tempDetail);  { ...add them }
        end;
     end;

{ add directory name and path to RemoteDirCombo }
     with FTPForm.RemoteDirCombo do
     begin
     { we let 32 lines in the history }
         if Items.Count >= 32 then Items.Delete(0);
         if Items.IndexOf(CurrentRemPath) = -1 then
             ItemIndex:= Items.Add(CurrentRemPath); { add path }
     end;

{ free directory manager }
     tempDirList.Free;
{ sort items }
     FTPForm.RemoteList.CustomSort(@CustomSortProc, 0);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.ConnectToRemoteHost;
const PreConMsg = 'Please wait, YAFTP is now resolving local and remote ' +
                  'host...';
var numOfRetries : word;    { guess wha? }
    connectRes   : byte;    { connect-attempt result }
    retries      : word;    { number of retries if server unreachable }
begin
    ControlIsFree:= false;
{ save login info }
    RemoteSite:= FTPForm.SiteComboBox.Text;
    if FTPForm.AnonLoginBox.Checked then
    begin
       UserName := 'anonymous';
       Password := SetupForm.SetupData.sEmailAddr
    end
    else
    begin
       UserName  := FTPForm.UserNameEdit.Text;
       Password  := FTPForm.PasswordEdit.Text;
    end;

{ check login data before connect; if it's missing, just simply exit }
    canContinue := true;
    if RemoteSite = '' then ShowError('Remote site not specified!',canContinue);
    if UserName   = '' then ShowError('No username specified!', canContinue);
    if Password   = '' then ShowError('No password specified!', canContinue);
    if IsCorrectNumber(1, 65535, 'FTP services are on port 21 (most times).',
                       FTPForm.PortEdit.Text)
    then
       RemotePort:= StrToInt(FTPForm.PortEdit.Text)
    else
       canContinue:= false;
    if not canContinue then exit;  { exit if there are errors }

{ show user what we're doing }
     FTPFTools.AddToCommArea(PreConMsg, caMisc);

     Application.ProcessMessages;

{ try to create FTPSession }
    try
       XFTPSession:= TFTPSession.Create(SetupForm.SetupData.sConnectTO,
                                        SetupForm.SetupData.sWelcomeTO,
                                        SetupForm.SetupData.sReplyTO, FTPForm)
    except
       on E:EFCannotGetLocalIP do begin
          AddToCommArea('ERROR   > Critical: ' + E.Message, caError);
          canContinue:= false;
       end;
    end;
    if not canContinue then exit;

{ show local host information }
    XFTPSession.GetLocalHost(localName, localIP, localIPDot);
    AddToCommArea(Format('INFO    > Local host is "%s", IP: %s',
                         [localName,localIPDot]), caLocal);

{ if instanting was successful, bind 'event handlers' }
    XFTPSession.OnCConConnected   := OnCConConnected;
    XFTPSession.OnCConCommand     := OnCConCommand;
    XFTPSession.OnServerReply     := OnServerReply;
    XFTPSession.OnCConClosed      := OnCConClosed;
    XFTPSession.OnCConError       := OnCConError;
    XFTPSession.OnDConConnected   := OnDConConnected;
    XFTPSession.OnDConClosed      := OnDConClosed;
    XFTPSession.OnDConError       := OnDConError;

{ disable unwanted buttons }
    FTPForm.ConnectB.Enabled:= false; FTPForm.DisconnectB.Enabled:= true;
    FTPForm.RemoteDirCombo.Clear;

{ try to connect to server }
    numOfRetries:= 1; retries:= SetupForm.SetupData.sRetSrvUnr;
    RetryOperation:= true;
    repeat
        AddToCommArea(Format('INFO    > Trying to connect "%s" to port %d... ' +
                             'attempt %d.', [remoteSite, remotePort,
                             numOfRetries]), caLocal);
        if FTPForm.AnonLoginBox.Checked then
           AddToCommArea('INFO    > Anonymous login, entered user name and ' +
                         'password ignored.', caLocal);

    { try to connect }
        connectRes:= XFTPSession.ConnectToServer(remoteSite, remotePort);
    { if there are retry(es) left, wait }
        if (connectRes <> smNO_ERROR) and (numOfRetries <= retries) then
           Delay(SetupForm.SetupData.sDelayBRet * 1000);

        inc(numOfRetries);

    { quit if we have to quit }
        if Application.Terminated then exit;
    until (connectRes = smNO_ERROR) or (numOfRetries > retries) or
          not canContinue;

    RetryOperation:= false;

{ if operation cancelled... }
    if not CanContinue then begin
       AddToCommArea('Connecting cancelled.', caMisc);
       FTPForm.ConnectB.Enabled:= true; FTPForm.DisconnectB.Enabled:= false;
       exit;
    end;

{ if there was an error... }
    if connectRes <> smNO_ERROR then begin
       AddToCommArea(Format('ERROR   > Couldn''t connect after %d retries.',
                     [retries]), caError);
       FTPForm.ConnectB.Enabled:= true; FTPForm.DisconnectB.Enabled:= false;
    end;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.Login;
var numOfRetries : word;    { guess wha? }
    loginRes     : byte;    { login-attempt result }
    retries      : word;    { number of retries if login failed }
begin
    ControlIsFree:= false;
{ try to connect to server }
    numOfRetries:= 1; retries:= SetupForm.SetupData.sRetLogFail;
    RetryOperation:= true;
    repeat
    { if not connected, connect first }
        if not ControlConnected then
        begin
            ConnectToRemoteHost;
            RetryOperation:= true { ConnectToRemoteHost set it to FALSE }
        end;

    { show info }
        AddToCommArea(Format('INFO    > Trying to login, attempt %d...',
                      [numOfRetries]), caLocal);
    { try to login }
        loginRes:= XFTPSession.Login(UserName, Password);
    { if there are retry(es) left, wait }
        if (loginRes = smLOGIN_FAILED) and (numOfRetries <= retries) then
            Delay(SetupForm.SetupData.sDelayBRet * 1000);

    { shut down FTPSession to create new one by ConnectToRemoteHost() }
        if (loginRes = smLOGIN_FAILED) then XFTPSession.Free;

    { always increment number of attempts }
        inc(numOfRetries);

    { quit if we have to quit }
        if Application.Terminated then exit;

    until (loginRes = smNO_ERROR) or (numOfRetries > retries) or
          not CanContinue;

    RetryOperation:= false;

{ if operation cancelled... }
    if not CanContinue then begin
       AddToCommArea('Login cancelled, control connection closed.', caMisc);
       FTPForm.ConnectB.Enabled:= true; FTPForm.DisconnectB.Enabled:= false;
       exit;
    end;

{ if there was an error... }
    if loginRes <> smNO_ERROR then begin
       AddToCommArea(Format('ERROR   > Couldn''t login after %d retries.',
                     [retries]), caError);
       FTPForm.ConnectB.Enabled:= true; FTPForm.DisconnectB.Enabled:= false;
       CanContinue:= false;
       exit;
    end;
end;

{------------------------------------------------------------------------------}

function TFTPFormTools.CanResumeDownloads: boolean;
begin
    if not Assigned(XFTPSession) then exit;
{ try to send REST 100 }
    Result:= false;
    if XFTPSession.RestartAt(100) = smNO_ERROR then
    begin
       XFTPSession.RestartAt(0);
    AddToCommArea('INFO    > This server allow you to resume broken downloads.',
                  caLocal);
       Result:= true;
    end
    else
      AddToCommArea('INFO    > This server doesn''t support resuming broken ' +
                    'downloads', caLocal);
end;

{------------------------------------------------------------------------------}

function TFTPFormTools.GetCurrentPath: string;
begin
    FTPForm.NOOPTimer.Enabled:= false;
{ get current path }
    Result:= XFTPSession.GetCurrentPath;
    if Result = '' then
    begin
      AddToCommArea('ERROR   > Critical: Couldn''t determine current path!',
                    caError);
      CanContinue:= false;
    end;
    FTPForm.NOOPTimer.Enabled:= true;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.LoadDir(ForceReload: boolean);
var currentPath : string;  { current remote path }
    cacheReply  : word;    { reply of cache manager }
    cacheFile   : string;  { filename in the cache }
    cached      : boolean; { if cached directory, it's TRUE }
    getDirReply : word;    { reply of GetDirList() }
begin
    FTPForm.NOOPTimer.Enabled:= false;

{ don't let user to click to RemoteList while downloading...}
    FTPForm.RemoteList.OnDblClick:= nil;  { nice solution :)}
    Application.ProcessMessages;

{ try to get remote path }
    currentPath:= XFTPSession.GetCurrentPath;
    currentRemPath:= CurrentPath;  { save it for external access, too. }

    if not CanContinue then begin
      AddToCommArea('ERROR   > Couldn''t get remote directory list.', caError);
      exit;
    end;

{ cache handling }
    if SetupForm.SetupData.sWantCache then
    begin
    { look for directory at cache }
       cacheReply:= XFTPCache.LoadFromCache(UserName, currentPath, cacheFile);

    { if ForceLoad, rewrite old cache entry }
       if (cacheReply in [cmNO_ERROR, cmEXPIRED])then
       begin
          cached:= true;
          cacheFile:= FTPForm.CacheDir + CacheFile;
          XFTPCache.RefreshLastItem;
       end
    { ... normal cache operations... }
       else
       { load from cache if possible }
          if ((cacheReply = cmEXPIRED) and (SetupForm.SetupData.sExpDays = 0))
              or (cacheReply = cmNO_ERROR) then
          begin
              cacheFile:= FTPForm.CacheDir + CacheFile;
              cached:= true;
          end
       { if we have to download it first...}
          else
          begin
              cached:= false;
              cacheFile:= XFTPCache.RegisterEntry(UserName, currentPath);
              if cacheFile = '' then { if register wasn't successful...}
              begin  { use system temporary path }
                 AddToCommArea('ERROR   > Couldn''t write cache.', caError);
                 cacheFile:= SysTempPath + '\yaftp_dirfile.temp'
              end
              else { add full path }
                 cacheFile:= FTPForm.CacheDir + cacheFile;
          end;
    end
{ get dirlist without using cache }
    else
       cacheFile:= SysTempPath + 'yaftp_dirfile.temp';

{ if not a cache directory in 'cacheFile', download list }
    if not cached or ForceReload then
    begin
    { set operation flag }
       DataOperation:= doList;
    { try receive remote directory list }
       if Assigned(XFTPSession) then
          getDirReply:= XFTPSession.GetDirList(cacheFile);
       if getDirReply <> smNO_ERROR then
          if getDirReply <> smTERMINATE then
          begin
            AddToCommArea('ERROR   > GetDirList(): ' + smErrorDesc[getDirReply],
                          caError);
            canContinue:= false;
            exit;
          end
          else
            AddToCommArea('Terminated.', caMisc)
       else
           AddToCommArea('Receiving directory list successfully finished.',
                         caMisc);
    end
    else
       AddToCommArea('Cached directory list reloaded from cache.', caMisc);

{ enable clicking on RemoteList,}
    FTPForm.RemoteList.OnDblClick:= FTPForm.RemoteListDblClick;
    FTPForm.NOOPTimer.Enabled:= true;

{ refresh remote browser }
    RefreshRemoteBrowser(cacheFile);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.Download(OverWrite: boolean; RemoteFile, RemoteSize,
                                 LocalFile, LocalSize: string);
var downReply : byte;   { reply of DownloadFile() }
    typeC     : char;   { transfer type }
    fileInfo  : string; { file information }
begin
    FTPForm.NOOPTimer.Enabled:= false;
{ whassup? Wouldn't you set this flag...? }
    DataOperation:= doDownload;

{ make file info string }
    fileInfo:= RemoteFile + ''', size ' + RemoteSize + ' bytes, mode ';

{ determine transfer type }
    if FTPForm.BinaryB.Down then
       begin typeC := 'I'; fileInfo:= fileInfo + 'binary' end;
    if FTPForm.ASCIIB.Down then
       begin typeC := 'A'; fileInfo:= fileInfo + 'ASCII' end;

{ show what we're doin'...}
    AddToCommArea('INFO    > Download ''' + fileInfo + '.', caLocal);

{ setup and show TransferForm }
    FTPForm.Enabled:= false;
    TransferForm.SetTransferInfo(true, typeC = 'I');
    TransferForm.SetFileInfo(LocalFile, RemoteFile, StrToInt(RemoteSize),
                             StrToInt(LocalSize));
    TransferForm.Show;
    TransferForm.Connecting;

{ download file }
    downReply:= XFTPSession.DownloadFile(LocalFile, RemoteFile, typeC,
                                         not OverWrite);
{ how it done? }
    if downReply <> smNO_ERROR then
       AddToCommArea('ERROR   > Download failed: ' + smErrorDesc[downReply],
                     caError)
    else
       AddToCommArea('Download successfully finished :)', caMisc);

{ close TransferForm }
    TransferForm.CloseForm;
    FTPForm.Enabled:= true;

{ refresh browsers }
    RefreshLocalBrowser;  { local }
    LoadDir(true);        { remote, force reload remote directory }
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.Upload(OverWrite: boolean; RemoteFile, RemoteSize,
                               LocalFile, LocalSize: string);
var upReply   : byte;   { reply of UploadFile() }
    typeC     : char;   { transfer type }
    fileInfo  : string; { file information }
begin
    FTPForm.NOOPTimer.Enabled:= false;
{ whassup? Wouldn't you set this flag...? }
    DataOperation:= doUpload;

{ make file info string }
    fileInfo:= LocalFile + ''', size ' + LocalSize + ' bytes, mode ';

{ determine transfer type }
    if FTPForm.BinaryB.Down then
       begin typeC := 'I'; fileInfo:= fileInfo + 'binary' end;
    if FTPForm.ASCIIB.Down then
       begin typeC := 'A'; fileInfo:= fileInfo + 'ASCII' end;

{ show what we're doin'...}
    AddToCommArea('INFO    > Upload ''' + fileInfo + '.', caLocal);

{ setup and show TransferForm }
    FTPForm.Enabled:= false;
    TransferForm.SetTransferInfo(false, typeC = 'I');
    TransferForm.SetFileInfo(LocalFile, RemoteFile, StrToInt(LocalSize),
                             StrToInt(RemoteSize));
    TransferForm.Show;
    TransferForm.Connecting;

{ upload file }
    upReply:= XFTPSession.UploadFile(LocalFile, RemoteFile, typeC,
                                     StrToInt(RemoteSize));

{ how it done? }
    if upReply <> smNO_ERROR then
       AddToCommArea('ERROR   > Upload failed: ' + smErrorDesc[upReply],
                     caError)
    else
       AddToCommArea('Upload successfully finished :)', caMisc);

{ close TransferForm }
    TransferForm.CloseForm;
    FTPForm.Enabled:= true;

{ refresh remote browser }
    LoadDir(true);        { remote, force reload remote directory }
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.SaveInitSettings;
var regIni: TRegIniFile;  { registry's INI-file emulator }
    x     : byte;         { cycle variable }
begin
{ create regini... }
    regIni:= TRegIniFile.Create('Yet Another FTP Client Settings');

{ save screen settings if it's not maximized or minimized }
    regIni.WriteInteger('SCREEN', 'Left',   FTPForm.Left);
    regIni.WriteInteger('SCREEN', 'Top',    FTPForm.Top);
    regIni.WriteInteger('SCREEN', 'Width',  FTPForm.Width);
    regIni.WriteInteger('SCREEN', 'Height', FTPForm.Height);
    regIni.WriteInteger('SCREEN', 'WindowState', integer(FTPForm.WindowState));

{ save SetupForm's data }
    regIni.WriteString ('MISC', 'UserEmail',   SetupForm.SetupData.sEmailAddr);
    regIni.WriteBool   ('MISC', 'WantCache',   SetupForm.SetupData.sWantCache);
    regIni.WriteBool   ('MISC', 'GCStartup',   SetupForm.SetupData.sGCStartup);
    regIni.WriteInteger('MISC', 'ExpDays',     SetupForm.SetupData.sExpDays);
    regIni.WriteInteger('MISC', 'ConnectTO',   SetupForm.SetupData.sConnectTO);
    regIni.WriteInteger('MISC', 'WelcomeTO',   SetupForm.SetupData.sWelcomeTO);
    regIni.WriteInteger('MISC', 'ReplyTO',     SetupForm.SetupData.sReplyTO);
    regIni.WriteInteger('MISC', 'NOOPSecs',    SetupForm.SetupData.sNOOPSecs);
    regIni.WriteInteger('MISC', 'RetrSrvUnr',  SetupForm.SetupData.sRetSrvUnr);
    regIni.WriteInteger('MISC', 'RetrLogFail', SetupForm.SetupData.sRetLogFail);
    regIni.WriteInteger('MISC', 'DelayBRet',   SetupForm.SetupData.sDelayBRet);
    regIni.WriteInteger('MISC', 'AutoTrn',     SetupForm.SetupData.sAutoTrn);

{ save site history }
    for x:=1 to 32 do
    { write site address if assigned }
        if x <= FTPForm.SiteComboBox.Items.Count then
           regIni.WriteString('SITE HISTORY', 'SiteLst' + IntToStr(x),
                              FTPForm.SiteComboBox.Items[x - 1])
        else
           regIni.WriteString('SITE HISTORY', 'SiteLst' + IntToStr(x), 'n/a');

{ free system registry regIni }
    regIni.Free;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.LoadInitSettings;
var regIni: TRegIniFile;  { registry's INI-file emulator }
    wState: TWindowState; { temporary window state }
    x     : byte;         { site history cyclevar }
    tempS : string;       { temporary string }
begin
{ create regini... }
    regIni:= TRegIniFile.Create('Yet Another FTP Client Settings');

{ load screen settings, default 639x479 }
    wState:= TWindowState(regIni.ReadInteger('SCREEN', 'WindowState',
                          integer(wsNormal)));

    if wState = wsNormal then
    begin
       FTPForm.Left   := regIni.ReadInteger('SCREEN', 'Left',  0);
       FTPForm.Top    := regIni.ReadInteger('SCREEN', 'Top',    0);
       FTPForm.Width  := regIni.ReadInteger('SCREEN', 'Width',  639);
       FTPForm.Height := regIni.ReadInteger('SCREEN', 'Height', 479);
    end
    else
       FTPForm.Position:= poDefault;

    FTPForm.WindowState:= wState;

{ load SetupForm's data; system settings}
    with SetupForm.SetupData do begin
       sEmailAddr := regIni.ReadString ('MISC', 'UserEmail',  '');
       sWantCache := regIni.ReadBool   ('MISC', 'WantCache',  false);
       sGCStartup := regIni.ReadBool   ('MISC', 'GCStartup',  false);
       sExpDays   := regIni.ReadInteger('MISC', 'ExpDays',    $ffff);
       sConnectTO := regIni.ReadInteger('MISC', 'ConnectTO',  0);
       sWelcomeTO := regIni.ReadInteger('MISC', 'WelcomeTO',  0);
       sReplyTO   := regIni.ReadInteger('MISC', 'ReplyTO',    0);
       sNOOPSecs  := regIni.ReadInteger('MISC', 'NOOPSecs',   0);
       sRetSrvUnr := regIni.ReadInteger('MISC', 'RetrSrvUnr', 0);
       sRetLogFail:= regIni.ReadInteger('MISC', 'RetrLogFail',0);
       sDelayBRet := regIni.ReadInteger('MISC', 'DelayBRet',  0);
       sAutoTrn   := regIni.ReadInteger('MISC', 'AutoTrn',    0);
    end;

{ if sExpDays has an invalid value ($ffff), read of settings is failed... }
    if SetupForm.SetupData.sExpDays = $ffff then
         SetupForm.ResetToDefaults;  { ... default data... }

{ save site history }
    for x:=1 to 32 do
    begin
    { read site addresses }
        tempS:= regIni.ReadString('SITE HISTORY','SiteLst'+IntToStr(x), 'n/a');
        if tempS <> 'n/a' then FTPForm.SiteComboBox.Items.Add(tempS);
    end;

{ free system registry regIni }
    regIni.Free;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.RunGC;
var tempFTPCache: TFTPCache;  { temporary FTP cache class for shit collecting }
    tempReturn  : byte;       { temporary return value of GarbageCollector() }
begin
    tempFTPCache:= TFTPCache.Create(FTPForm.CacheDir, '0.0.0.0',
                                    SetupForm.SetupData.sExpDays);
{ show collecting state }
    AddToCommArea('INFO    > Starting garbage collector in cache...', caLocal);

{ run garbage collector }
    tempReturn:= tempFTPCache.GarbageCollector;  { run garbage collector }

{ if no cache found... }
    if tempReturn = cmNO_CACHE then
    begin
       tempFTPCache.Free;
       exit;
    end;

{ if there was an error while emptying... }
    if tempReturn <> cmNO_ERROR then
       AddToCommArea('INFO    > Garbage collecting failed: ' +
                     cmMessages[tempReturn] + ' It''s sad :(', caError)
    else
       AddToCommArea('INFO    > Garbage collecting successfully finished :) ',
                     caLocal);

{ free FTP cache manager }
    tempFTPCache.Free;
end;
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ "Event handlers" ------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TFTPFormTools.OnCConConnected;
begin
    ControlConnected:= true;
    AddToCommArea('INFO    > Control connection has been built, login sequence'+
                  ' started...', caLocal);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnCConCommand;
begin
    AddToCommArea('COMMAND > ' + XFTPSession.GetLastCommand, caLocal);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnServerReply;
var addCount: longint;      { cycle variable }
begin
{ add each item }
    if not Assigned(XFTPSession) then Exit;
    for addCount:=0 to XFTPSession.GetServerReply.Count - 1 do
      AddToCommArea('SERVER  < ' + XFTPSession.GetServerReply[addCount],
                     caRemote);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnCConClosed;
begin
{ determine whether strange Connection Refused occur... }
    if ControlConnected then
       AddToCommArea('INFO    > Control connection closed.', caLocal)
    else
       AddToCommArea('ERROR   > Remote server refused connection request (you' +
                     ' may change port number!).', caError);

{ enable connect control if no retry-sequence is in progress }
    if not RetryOperation then
    begin
       FTPForm.ConnectB.Enabled:= true;
       FTPForm.DisconnectB.Enabled:= false;
       FTPForm.RemoteList.Visible:= false;
       FTPForm.RemoteList.Items.Clear;
       FTPForm.RemoteList.Visible:= true;
       FTPForm.RemoteDirCombo.Items.Clear;
    end;

    ControlConnected:= false;
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnCConError;
begin
    AddToCommArea('ERROR   > Control connection: ' +
                  XFTPSession.GetLastErrorText, caError);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnDConConnected;
var Msg: string;
begin
{ display startup info }
    case DataOperation of
       doList: Msg:= 'Receiving remote directory list. It may take a while, ' +
                     'be patient.';
       doDownload: begin
                     Msg:= 'Receiving file has been started.';
                     TransferForm.Transferring;
                   end;
       doUpload:   begin
                     Msg:= 'Sending file has been started.';
                     TransferForm.Transferring;
                   end;

    end;
    if Msg <> '' then AddToCommArea(Msg, caMisc);
    AddToCommArea('INFO    > Data connection has built, transferring...',
                   caLocal);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnDConClosed;
begin
{ if we're (were) at transfer, we have to close TransferForm and enable FTFF. }
    if (DataOperation = doDownload) or (DataOperation = doUpload) then
    begin
        TransferForm.CloseForm;
        FTPForm.Enabled:= true;
    end;

{ show event }
    AddToCommArea('INFO    > Data connection closed.', caLocal);
end;

{------------------------------------------------------------------------------}

procedure TFTPFormTools.OnDConError;
begin
    AddToCommArea('ERROR   > Data connection: ' + XFTPSession.GetLastErrorText,
                   caError);
end;

{------------------------------------------------------------------------------}

end.
